using System;
using System.Collections;
using System.Drawing;
using System.Windows.Forms;

namespace gov.va.med.vbecs.admin.GUI
{
	/// <summary>
	/// Utility UI class. 
	/// </summary>
	public class GuiUtility
	{
		private GuiUtility() {}

		/// <summary>
		/// Populates specified combo box from a given list. 
		/// </summary>
		/// <param name="comboBox">Combo box to populate.</param>
		/// <param name="sourceList">Source data list.</param>
		/// <param name="displayMember">Combo box display member for data list item.</param>
		public static void PopulateComboBoxFromList( ComboBox comboBox, IEnumerable sourceList, string displayMember )
		{			
			if( comboBox == null )
				throw( new ArgumentException( "comboBox" ) );

			if( sourceList == null )
				throw( new ArgumentNullException( "sourceList" ) );

			if( displayMember == null )
				throw( new ArgumentNullException( "displayMember" ) );

			comboBox.SuspendLayout();

			comboBox.SelectedItem = null; // this is needed, otherwise the item will remain selected.
			comboBox.Items.Clear();

			comboBox.DisplayMember = displayMember;

			foreach( object _item in sourceList )
				comboBox.Items.Add( _item );

			comboBox.ResumeLayout();
		}

		/// <summary>
		/// Sets minimum and maximum values for a specified <see cref="NumericUpDown"/> control.
		/// </summary>
		/// <param name="updControl">Control to set limits for.</param>
		/// <param name="minValue">Minimum value.</param>
		/// <param name="maxValue">Maximum value.</param>
		public static void SetNumericUpdownMinMax( NumericUpDown updControl, decimal minValue, decimal maxValue )
		{
			if( updControl == null )
				throw( new ArgumentNullException( "updControl" ) );

			updControl.Minimum = minValue;
			updControl.Maximum = maxValue;
		}

		/// <summary>
		/// Switches listview to list mode (no columns and headers) and sets a single message (single item).
		/// Also disables listview.
		/// </summary>
		/// <param name="listView">Control to set message on.</param>
		/// <param name="message">String message.</param>
		public static void SetListViewReadonlyErrorMessage( ListView listView, string message )
		{
			if( listView == null )
				throw( new ArgumentNullException( "listView" ) );

			listView.BeginUpdate();

			listView.View = View.List;
			listView.Enabled = false;
			listView.BackColor = SystemColors.Control;

			listView.Items.Clear();
			listView.Items.Add( Common.StUiNullConvert.From( message ) );

			listView.EndUpdate();
		}

		/// <summary>
		/// Reenables <see cref="ListView"/> that had an error message set with <see cref="SetListViewReadonlyErrorMessage"/>.
		/// </summary>
		/// <param name="listView"><see cref="ListView"/> to update.</param>
		public static void ReenableDetailedListView( ListView listView )
		{
			if( listView == null )
				throw( new ArgumentNullException( "listView" ) );

			listView.View = View.Details;
			listView.Enabled = true;
			listView.BackColor = SystemColors.Window;			
		}
	}
}
